import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:mighty_school/common/widget/custom_app_bar.dart';
import 'package:mighty_school/common/widget/custom_button.dart';
import 'package:mighty_school/common/widget/custom_contaner.dart';
import 'package:mighty_school/common/widget/custom_text_field.dart';
import 'package:mighty_school/common/widget/custom_dropdown.dart';
import 'package:mighty_school/feature/hostel_management/hostel/domain/model/hostel_body.dart';
import 'package:mighty_school/feature/hostel_management/hostel/logic/hostel_controller.dart';
import 'package:mighty_school/helper/responsive_helper.dart';
import 'package:mighty_school/util/dimensions.dart';

class AddNewHostelScreen extends StatefulWidget {
  final int? hostelId;
  const AddNewHostelScreen({super.key, this.hostelId});

  @override
  State<AddNewHostelScreen> createState() => _AddNewHostelScreenState();
}

class _AddNewHostelScreenState extends State<AddNewHostelScreen> {
  final TextEditingController nameController = TextEditingController();
  final TextEditingController addressController = TextEditingController();
  final TextEditingController phoneController = TextEditingController();
  final TextEditingController emailController = TextEditingController();
  final TextEditingController descriptionController = TextEditingController();
  final TextEditingController capacityController = TextEditingController();
  final GlobalKey<FormState> formKey = GlobalKey<FormState>();

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: CustomAppBar(
        title: widget.hostelId != null ? "edit_hostel".tr : "add_new_hostel".tr
      ),
      body: GetBuilder<HostelController>(
        builder: (hostelController) {
          return SingleChildScrollView(
            padding: const EdgeInsets.all(Dimensions.paddingSizeDefault),
            child: Form(
              key: formKey,
              child: CustomContainer(
                showShadow: ResponsiveHelper.isDesktop(context),
                child: Column(
                  children: [

                    
                    const SizedBox(height: Dimensions.paddingSizeDefault),
                    
                    // Name field
                    CustomTextField(
                      controller: nameController,
                      labelText: "hostel_name".tr,
                      isRequired: true,
                      validator: (value) {
                        if (value == null || value.isEmpty) {
                          return "please_enter_hostel_name".tr;
                        }
                        return null;
                      },
                    ),
                    
                    const SizedBox(height: Dimensions.paddingSizeDefault),
                    
                    // Address field
                    CustomTextField(
                      controller: addressController,
                      labelText: "address".tr,
                      isRequired: true,
                      maxLines: 3,
                      validator: (value) {
                        if (value == null || value.isEmpty) {
                          return "please_enter_address".tr;
                        }
                        return null;
                      },
                    ),
                    
                    const SizedBox(height: Dimensions.paddingSizeDefault),
                    
                    // Phone field
                    CustomTextField(
                      controller: phoneController,
                      labelText: "phone".tr,
                      isRequired: true,
                      validator: (value) {
                        if (value == null || value.isEmpty) {
                          return "please_enter_phone".tr;
                        }
                        return null;
                      },
                    ),
                    
                    const SizedBox(height: Dimensions.paddingSizeDefault),
                    
                    // Email field
                    CustomTextField(
                      controller: emailController,
                      labelText: "email".tr,
                      validator: (value) {
                        if (value != null && value.isNotEmpty && !GetUtils.isEmail(value)) {
                          return "please_enter_valid_email".tr;
                        }
                        return null;
                      },
                    ),
                    
                    const SizedBox(height: Dimensions.paddingSizeDefault),
                    
                    // Capacity field
                    CustomTextField(
                      controller: capacityController,
                      labelText: "capacity".tr,

                      isRequired: true,
                      validator: (value) {
                        if (value == null || value.isEmpty) {
                          return "please_enter_capacity".tr;
                        }
                        return null;
                      },
                    ),
                    
                    const SizedBox(height: Dimensions.paddingSizeDefault),
                    
                    // Status dropdown
                    CustomDropdown(
                      title: "status".tr,
                      selectedValue: hostelController.selectedStatus,
                      items: hostelController.statusList,
                      onChanged: (value) => hostelController.setSelectedStatus(value!),
                    ),
                    
                    const SizedBox(height: Dimensions.paddingSizeDefault),
                    
                    // Description field
                    CustomTextField(
                      controller: descriptionController,
                      labelText: "description".tr,
                      maxLines: 4,
                    ),
                    
                    const SizedBox(height: Dimensions.paddingSizeLarge),
                    
                    // Submit button
                    CustomButton(
                      text: widget.hostelId != null ? "update".tr : "add".tr,
                      onTap: () {
                        if (formKey.currentState!.validate()) {
                          HostelBody hostelBody = HostelBody(
                            name: nameController.text.trim(),
                            address: addressController.text.trim(),
                            phone: phoneController.text.trim(),
                            email: emailController.text.trim().isEmpty ? null : emailController.text.trim(),
                            description: descriptionController.text.trim().isEmpty ? null : descriptionController.text.trim(),
                            capacity: capacityController.text.trim(),
                            status: hostelController.selectedStatus,
                          );
                          
                          if (widget.hostelId != null) {
                            hostelController.updateHostel(widget.hostelId!, hostelBody);
                          } else {
                            hostelController.addNewHostel(hostelBody);
                          }
                        }
                      },
                    ),
                  ],
                ),
              ),
            ),
          );
        },
      ),
    );
  }
  
  @override
  void dispose() {
    nameController.dispose();
    addressController.dispose();
    phoneController.dispose();
    emailController.dispose();
    descriptionController.dispose();
    capacityController.dispose();
    super.dispose();
  }
}
