import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:mighty_school/common/widget/custom_contaner.dart';
import 'package:mighty_school/common/widget/custom_image.dart';
import 'package:mighty_school/common/widget/edit_delete_section.dart';
import 'package:mighty_school/common/widget/numbering_widget.dart';
import 'package:mighty_school/feature/hostel_management/hostel/domain/model/hostel_model.dart';
import 'package:mighty_school/feature/hostel_management/hostel/logic/hostel_controller.dart';
import 'package:mighty_school/feature/hostel_management/hostel/presentation/screens/add_new_hostel_screen.dart';
import 'package:mighty_school/helper/responsive_helper.dart';
import 'package:mighty_school/util/dimensions.dart';
import 'package:mighty_school/util/styles.dart';

class HostelItemWidget extends StatelessWidget {
  final HostelItem hostelItem;
  final int index;
  
  const HostelItemWidget({
    super.key, 
    required this.hostelItem, 
    required this.index
  });

  @override
  Widget build(BuildContext context) {
    return GetBuilder<HostelController>(
      builder: (hostelController) {
        return ResponsiveHelper.isDesktop(context) 
          ? _buildDesktopView(context, hostelController)
          : _buildMobileView(context, hostelController);
      },
    );
  }

  Widget _buildDesktopView(BuildContext context, HostelController hostelController) {
    return CustomContainer(
      child: Row(
        children: [
          // Numbering
          NumberingWidget(index: index),
          
          // Image
          const Expanded(
            flex: 1,
            child: CustomImage(
              image: '',
              height: 40,
              width: 40,
              fit: BoxFit.cover,
            ),
          ),
          
          // Name
          Expanded(
            flex: 2,
            child: Text(
              hostelItem.name ?? '',
              style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
              maxLines: 2,
              overflow: TextOverflow.ellipsis,
            ),
          ),
          
          // Address
          Expanded(
            flex: 2,
            child: Text(
              hostelItem.address ?? '',
              style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
              maxLines: 2,
              overflow: TextOverflow.ellipsis,
            ),
          ),
          
          // Phone
          Expanded(
            flex: 1,
            child: Text(
              hostelItem.phone ?? '',
              style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
            ),
          ),
          
          // Capacity
          Expanded(
            flex: 1,
            child: Text(
              hostelItem.capacity?.toString() ?? '0',
              style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
            ),
          ),
          
          // Status
          Expanded(
            flex: 1,
            child: Container(
              padding: const EdgeInsets.symmetric(
                horizontal: Dimensions.paddingSizeSmall,
                vertical: Dimensions.paddingSizeExtraSmall,
              ),
              decoration: BoxDecoration(
                color: hostelItem.status == 'Active' 
                  ? Colors.green.withValues(alpha: 0.1)
                  : Colors.red.withValues(alpha: 0.1),
                borderRadius: BorderRadius.circular(Dimensions.radiusSmall),
              ),
              child: Text(
                hostelItem.status ?? '',
                style: textRegular.copyWith(
                  fontSize: Dimensions.fontSizeSmall,
                  color: hostelItem.status == 'Active' ? Colors.green : Colors.red,
                ),
                textAlign: TextAlign.center,
              ),
            ),
          ),
          
          // Actions
          Expanded(
            flex: 1,
            child: EditDeleteSection(
              onEdit: () => Get.to(() => AddNewHostelScreen(hostelId: hostelItem.id)),
              onDelete: () => hostelController.deleteHostel(hostelItem.id!),
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildMobileView(BuildContext context, HostelController hostelController) {
    return CustomContainer(
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            children: [
              // Image
              const CustomImage(
                image:  '',
                height: 50,
                width: 50,
                fit: BoxFit.cover,
              ),
              
              const SizedBox(width: Dimensions.paddingSizeSmall),
              
              // Name and details
              Expanded(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Text(
                      hostelItem.name ?? '',
                      style: textMedium.copyWith(fontSize: Dimensions.fontSizeDefault),
                      maxLines: 1,
                      overflow: TextOverflow.ellipsis,
                    ),
                    
                    const SizedBox(height: Dimensions.paddingSizeExtraSmall),
                    
                    Text(
                      hostelItem.address ?? '',
                      style: textRegular.copyWith(
                        fontSize: Dimensions.fontSizeSmall,
                        color: Theme.of(context).hintColor,
                      ),
                      maxLines: 2,
                      overflow: TextOverflow.ellipsis,
                    ),
                  ],
                ),
              ),
              
              // Status
              Container(
                padding: const EdgeInsets.symmetric(
                  horizontal: Dimensions.paddingSizeSmall,
                  vertical: Dimensions.paddingSizeExtraSmall,
                ),
                decoration: BoxDecoration(
                  color: hostelItem.status == 'Active' 
                    ? Colors.green.withValues(alpha: 0.1)
                    : Colors.red.withValues(alpha: 0.1),
                  borderRadius: BorderRadius.circular(Dimensions.radiusSmall),
                ),
                child: Text(
                  hostelItem.status ?? '',
                  style: textRegular.copyWith(
                    fontSize: Dimensions.fontSizeSmall,
                    color: hostelItem.status == 'Active' ? Colors.green : Colors.red,
                  ),
                ),
              ),
            ],
          ),
          
          const SizedBox(height: Dimensions.paddingSizeSmall),
          
          // Additional info
          Row(
            children: [
              Expanded(
                child: Text(
                  "${'phone'.tr}: ${hostelItem.phone ?? 'N/A'}",
                  style: textRegular.copyWith(fontSize: Dimensions.fontSizeSmall),
                ),
              ),
              
              Expanded(
                child: Text(
                  "${'capacity'.tr}: ${hostelItem.capacity?.toString() ?? '0'}",
                  style: textRegular.copyWith(fontSize: Dimensions.fontSizeSmall),
                ),
              ),
            ],
          ),
          
          const SizedBox(height: Dimensions.paddingSizeSmall),
          
          // Actions
          EditDeleteSection(
            onEdit: () => Get.to(() => AddNewHostelScreen(hostelId: hostelItem.id)),
            onDelete: () => hostelController.deleteHostel(hostelItem.id!),
          ),
        ],
      ),
    );
  }
}
