import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:mighty_school/common/widget/custom_contaner.dart';
import 'package:mighty_school/common/widget/custom_image.dart';
import 'package:mighty_school/common/widget/edit_delete_section.dart';
import 'package:mighty_school/common/widget/numbering_widget.dart';
import 'package:mighty_school/feature/hostel_management/hostel_members/domain/model/hostel_member_model.dart';
import 'package:mighty_school/feature/hostel_management/hostel_members/logic/hostel_members_controller.dart';
import 'package:mighty_school/feature/hostel_management/hostel_members/presentation/screens/add_new_hostel_member_screen.dart';
import 'package:mighty_school/helper/responsive_helper.dart';
import 'package:mighty_school/util/dimensions.dart';
import 'package:mighty_school/util/styles.dart';

class HostelMemberItemWidget extends StatelessWidget {
  final HostelMemberItem memberItem;
  final int index;
  
  const HostelMemberItemWidget({
    super.key, 
    required this.memberItem, 
    required this.index
  });

  @override
  Widget build(BuildContext context) {
    return GetBuilder<HostelMembersController>(
      builder: (memberController) {
        return ResponsiveHelper.isDesktop(context) 
          ? _buildDesktopView(context, memberController)
          : _buildMobileView(context, memberController);
      },
    );
  }

  Widget _buildDesktopView(BuildContext context, HostelMembersController memberController) {
    return CustomContainer(
      child: Row(
        children: [
          // Numbering
          NumberingWidget(index: index),
          
          // Student
          Expanded(
            flex: 2,
            child: Row(
              children: [
                CustomImage(
                  image: memberItem.student?.image ?? '',
                  height: 30,
                  width: 30,
                  fit: BoxFit.cover,
                ),
                const SizedBox(width: Dimensions.paddingSizeSmall),
                Expanded(
                  child: Text(
                    "${memberItem.student?.firstName ?? ''} ${memberItem.student?.lastName ?? ''}".trim(),
                    style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
                    maxLines: 1,
                    overflow: TextOverflow.ellipsis,
                  ),
                ),
              ],
            ),
          ),
          
          // Hostel
          Expanded(
            flex: 2,
            child: Text(
              memberItem.hostel?.name ?? '',
              style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
              maxLines: 1,
              overflow: TextOverflow.ellipsis,
            ),
          ),
          
          // Join Date
          Expanded(
            flex: 1,
            child: Text(
              memberItem.joinDate ?? '',
              style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
            ),
          ),
          
          // Leave Date
          Expanded(
            flex: 1,
            child: Text(
              memberItem.leaveDate ?? 'N/A',
              style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
            ),
          ),
          
          // Status
          Expanded(
            flex: 1,
            child: Container(
              padding: const EdgeInsets.symmetric(
                horizontal: Dimensions.paddingSizeSmall,
                vertical: Dimensions.paddingSizeExtraSmall,
              ),
              decoration: BoxDecoration(
                color: memberItem.status == 'Active' 
                  ? Colors.green.withValues(alpha: 0.1)
                  : Colors.red.withValues(alpha: 0.1),
                borderRadius: BorderRadius.circular(Dimensions.radiusSmall),
              ),
              child: Text(
                memberItem.status ?? '',
                style: textRegular.copyWith(
                  fontSize: Dimensions.fontSizeSmall,
                  color: memberItem.status == 'Active' ? Colors.green : Colors.red,
                ),
                textAlign: TextAlign.center,
              ),
            ),
          ),
          
          // Actions
          Expanded(
            flex: 1,
            child: EditDeleteSection(
              onEdit: () => Get.to(() => AddNewHostelMemberScreen(memberId: memberItem.id)),
              onDelete: () => memberController.deleteHostelMember(memberItem.id!),
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildMobileView(BuildContext context, HostelMembersController memberController) {
    return CustomContainer(
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            children: [
              // Student image
              CustomImage(
                image: memberItem.student?.image ?? '',
                height: 40,
                width: 40,
                fit: BoxFit.cover,
              ),
              
              const SizedBox(width: Dimensions.paddingSizeSmall),
              
              // Student name and hostel
              Expanded(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Text(
                      "${memberItem.student?.firstName ?? ''} ${memberItem.student?.lastName ?? ''}".trim(),
                      style: textMedium.copyWith(fontSize: Dimensions.fontSizeDefault),
                      maxLines: 1,
                      overflow: TextOverflow.ellipsis,
                    ),
                    
                    const SizedBox(height: Dimensions.paddingSizeExtraSmall),
                    
                    Text(
                      memberItem.hostel?.name ?? '',
                      style: textRegular.copyWith(
                        fontSize: Dimensions.fontSizeSmall,
                        color: Theme.of(context).hintColor,
                      ),
                      maxLines: 1,
                      overflow: TextOverflow.ellipsis,
                    ),
                  ],
                ),
              ),
              
              // Status
              Container(
                padding: const EdgeInsets.symmetric(
                  horizontal: Dimensions.paddingSizeSmall,
                  vertical: Dimensions.paddingSizeExtraSmall,
                ),
                decoration: BoxDecoration(
                  color: memberItem.status == 'Active' 
                    ? Colors.green.withValues(alpha: 0.1)
                    : Colors.red.withValues(alpha: 0.1),
                  borderRadius: BorderRadius.circular(Dimensions.radiusSmall),
                ),
                child: Text(
                  memberItem.status ?? '',
                  style: textRegular.copyWith(
                    fontSize: Dimensions.fontSizeSmall,
                    color: memberItem.status == 'Active' ? Colors.green : Colors.red,
                  ),
                ),
              ),
            ],
          ),
          
          const SizedBox(height: Dimensions.paddingSizeSmall),
          
          // Dates
          Row(
            children: [
              Expanded(
                child: Text(
                  "${'join_date'.tr}: ${memberItem.joinDate ?? 'N/A'}",
                  style: textRegular.copyWith(fontSize: Dimensions.fontSizeSmall),
                ),
              ),
              
              Expanded(
                child: Text(
                  "${'leave_date'.tr}: ${memberItem.leaveDate ?? 'N/A'}",
                  style: textRegular.copyWith(fontSize: Dimensions.fontSizeSmall),
                ),
              ),
            ],
          ),
          
          const SizedBox(height: Dimensions.paddingSizeSmall),
          
          // Actions
          EditDeleteSection(
            onEdit: () => Get.to(() => AddNewHostelMemberScreen(memberId: memberItem.id)),
            onDelete: () => memberController.deleteHostelMember(memberItem.id!),
          ),
        ],
      ),
    );
  }
}
