import 'package:get/get.dart';
import 'package:mighty_school/api_handle/api_checker.dart';
import 'package:mighty_school/common/widget/custom_snackbar.dart';
import 'package:mighty_school/feature/hostel_management/hostel_rooms/domain/model/hostel_room_body.dart';
import 'package:mighty_school/feature/hostel_management/hostel_rooms/domain/model/hostel_room_model.dart';
import 'package:mighty_school/feature/hostel_management/hostel_rooms/domain/repository/hostel_rooms_repository.dart';

class HostelRoomsController extends GetxController implements GetxService {
  final HostelRoomsRepository hostelRoomsRepository;
  HostelRoomsController({required this.hostelRoomsRepository});

  bool isLoading = false;
  HostelRoomModel? hostelRoomModel;

  // Get hostel rooms list
  Future<void> getHostelRoomsList({int page = 1}) async {
    if (page == 1) {
      isLoading = true;
      update();
    }

    Response? response = await hostelRoomsRepository.getHostelRoomsList(page);
    if (response?.statusCode == 200) {
      if (page == 1) {
        hostelRoomModel = HostelRoomModel.fromJson(response?.body);
      } else {
        hostelRoomModel?.data?.data?.addAll(HostelRoomModel.fromJson(response?.body).data!.data!);
        hostelRoomModel?.data?.total = HostelRoomModel.fromJson(response?.body).data!.total;
        hostelRoomModel?.data?.currentPage = HostelRoomModel.fromJson(response?.body).data!.currentPage;
      }
      isLoading = false;
    } else {
      isLoading = false;
      ApiChecker.checkApi(response!);
    }
    update();
  }

  // Add new hostel room
  Future<void> addNewHostelRoom(HostelRoomBody roomBody) async {
    isLoading = true;
    update();

    Response? response = await hostelRoomsRepository.addNewHostelRoom(roomBody);
    if (response?.statusCode == 200 || response?.statusCode == 201) {
      isLoading = false;
      Get.back();
      showCustomSnackBar('hostel_room_added_successfully'.tr, isError: false);
      getHostelRoomsList();
    } else {
      isLoading = false;
      ApiChecker.checkApi(response!);
    }
    update();
  }

  // Update hostel room
  Future<void> updateHostelRoom(int id, HostelRoomBody roomBody) async {
    isLoading = true;
    update();

    Response? response = await hostelRoomsRepository.updateHostelRoom(id, roomBody);
    if (response?.statusCode == 200) {
      isLoading = false;
      Get.back();
      showCustomSnackBar('hostel_room_updated_successfully'.tr, isError: false);
      getHostelRoomsList();
    } else {
      isLoading = false;
      ApiChecker.checkApi(response!);
    }
    update();
  }

  // Delete hostel room
  Future<void> deleteHostelRoom(int id) async {
    Response? response = await hostelRoomsRepository.deleteHostelRoom(id);
    if (response?.statusCode == 200) {
      showCustomSnackBar('hostel_room_deleted_successfully'.tr, isError: false);
      getHostelRoomsList();
    } else {
      ApiChecker.checkApi(response!);
    }
    update();
  }

  // Room types
  List<String> roomTypeList = ["Single", "Double", "Triple", "Dormitory"];
  String selectedRoomType = "Single";

  void setSelectedRoomType(String roomType) {
    selectedRoomType = roomType;
    update();
  }

  // Status list
  List<String> statusList = ["Available", "Occupied", "Maintenance"];
  String selectedStatus = "Available";

  void setSelectedStatus(String status) {
    selectedStatus = status;
    update();
  }
}

