import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:mighty_school/common/widget/custom_contaner.dart';
import 'package:mighty_school/common/widget/edit_delete_section.dart';
import 'package:mighty_school/common/widget/numbering_widget.dart';
import 'package:mighty_school/feature/hostel_management/hostel_rooms/domain/model/hostel_room_model.dart';
import 'package:mighty_school/feature/hostel_management/hostel_rooms/logic/hostel_rooms_controller.dart';
import 'package:mighty_school/feature/hostel_management/hostel_rooms/presentation/screens/add_new_hostel_room_screen.dart';
import 'package:mighty_school/helper/responsive_helper.dart';
import 'package:mighty_school/util/dimensions.dart';
import 'package:mighty_school/util/styles.dart';

class HostelRoomItemWidget extends StatelessWidget {
  final HostelRoomItem roomItem;
  final int index;

  const HostelRoomItemWidget({
    super.key,
    required this.roomItem,
    required this.index
  });

  @override
  Widget build(BuildContext context) {
    return GetBuilder<HostelRoomsController>(
      builder: (roomsController) {
        return ResponsiveHelper.isDesktop(context)
          ? _buildDesktopView(context, roomsController)
          : _buildMobileView(context, roomsController);
      },
    );
  }

  Widget _buildDesktopView(BuildContext context, HostelRoomsController roomsController) {
    return CustomContainer(
      child: Row(
        children: [
          // Numbering
          NumberingWidget(index: index),

          // Room Number
          Expanded(
            flex: 2,
            child: Text(
              roomItem.roomNumber ?? '',
              style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
              maxLines: 1,
              overflow: TextOverflow.ellipsis,
            ),
          ),

          // Hostel Name
          Expanded(
            flex: 2,
            child: Text(
              roomItem.hostel?.name ?? '',
              style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
              maxLines: 2,
              overflow: TextOverflow.ellipsis,
            ),
          ),

          // Room Type
          Expanded(
            flex: 1,
            child: Text(
              roomItem.roomType ?? '',
              style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
            ),
          ),

          // Capacity
          Expanded(
            flex: 1,
            child: Text(
              roomItem.capacity?.toString() ?? '0',
              style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
            ),
          ),

          // Occupancy
          Expanded(
            flex: 1,
            child: Text(
              '${roomItem.currentOccupancy ?? 0}/${roomItem.capacity ?? 0}',
              style: textRegular.copyWith(fontSize: Dimensions.fontSizeDefault),
            ),
          ),

          // Status
          Expanded(
            flex: 1,
            child: Container(
              padding: const EdgeInsets.symmetric(
                horizontal: Dimensions.paddingSizeSmall,
                vertical: Dimensions.paddingSizeExtraSmall,
              ),
              decoration: BoxDecoration(
                color: roomItem.status == 'Available'
                  ? Colors.green.withValues(alpha: 0.1)
                  : roomItem.status == 'Occupied'
                    ? Colors.orange.withValues(alpha: 0.1)
                    : Colors.red.withValues(alpha: 0.1),
                borderRadius: BorderRadius.circular(Dimensions.radiusSmall),
              ),
              child: Text(
                roomItem.status ?? '',
                style: textRegular.copyWith(
                  fontSize: Dimensions.fontSizeSmall,
                  color: roomItem.status == 'Available'
                    ? Colors.green
                    : roomItem.status == 'Occupied'
                      ? Colors.orange
                      : Colors.red,
                ),
                textAlign: TextAlign.center,
              ),
            ),
          ),

          // Actions
          Expanded(
            flex: 1,
            child: EditDeleteSection(
              onEdit: () => Get.to(() => AddNewHostelRoomScreen(roomId: roomItem.id)),
              onDelete: () => roomsController.deleteHostelRoom(roomItem.id!),
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildMobileView(BuildContext context, HostelRoomsController roomsController) {
    return CustomContainer(
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            children: [
              // Room info
              Expanded(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Text(
                      roomItem.roomNumber ?? '',
                      style: textMedium.copyWith(fontSize: Dimensions.fontSizeDefault),
                      maxLines: 1,
                      overflow: TextOverflow.ellipsis,
                    ),

                    const SizedBox(height: Dimensions.paddingSizeExtraSmall),

                    Text(
                      roomItem.hostel?.name ?? '',
                      style: textRegular.copyWith(
                        fontSize: Dimensions.fontSizeSmall,
                        color: Theme.of(context).hintColor,
                      ),
                      maxLines: 2,
                      overflow: TextOverflow.ellipsis,
                    ),
                  ],
                ),
              ),

              // Status
              Container(
                padding: const EdgeInsets.symmetric(
                  horizontal: Dimensions.paddingSizeSmall,
                  vertical: Dimensions.paddingSizeExtraSmall,
                ),
                decoration: BoxDecoration(
                  color: roomItem.status == 'Available'
                    ? Colors.green.withValues(alpha: 0.1)
                    : roomItem.status == 'Occupied'
                      ? Colors.orange.withValues(alpha: 0.1)
                      : Colors.red.withValues(alpha: 0.1),
                  borderRadius: BorderRadius.circular(Dimensions.radiusSmall),
                ),
                child: Text(
                  roomItem.status ?? '',
                  style: textRegular.copyWith(
                    fontSize: Dimensions.fontSizeSmall,
                    color: roomItem.status == 'Available'
                      ? Colors.green
                      : roomItem.status == 'Occupied'
                        ? Colors.orange
                        : Colors.red,
                  ),
                ),
              ),
            ],
          ),

          const SizedBox(height: Dimensions.paddingSizeSmall),

          // Additional info
          Row(
            children: [
              Expanded(
                child: Text(
                  "${'room_type'.tr}: ${roomItem.roomType ?? 'N/A'}",
                  style: textRegular.copyWith(fontSize: Dimensions.fontSizeSmall),
                ),
              ),

              Expanded(
                child: Text(
                  "${'capacity'.tr}: ${roomItem.capacity?.toString() ?? '0'}",
                  style: textRegular.copyWith(fontSize: Dimensions.fontSizeSmall),
                ),
              ),
            ],
          ),

          const SizedBox(height: Dimensions.paddingSizeExtraSmall),

          Text(
            "${'occupancy'.tr}: ${roomItem.currentOccupancy ?? 0}/${roomItem.capacity ?? 0}",
            style: textRegular.copyWith(fontSize: Dimensions.fontSizeSmall),
          ),

          const SizedBox(height: Dimensions.paddingSizeSmall),

          // Actions
          EditDeleteSection(
            onEdit: () => Get.to(() => AddNewHostelRoomScreen(roomId: roomItem.id)),
            onDelete: () => roomsController.deleteHostelRoom(roomItem.id!),
          ),
        ],
      ),
    );
  }
}
